package ttsolver.constraint;

import ifs.model.*;
import java.util.Set;
import ttsolver.model.*;

/**
 * Resource constraint.
 * <br>
 * Classes using this resource can not overlap in time. All classes associated with this
 * resource are considered to use this resource.
 * 
 * @author <a href="mailto:muller@ktiml.mff.cuni.cz">Tomas Muller</a>
 * @version 1.0
 */

public class ResourceConstraint extends Constraint {
    private static org.apache.log4j.Logger sLogger = org.apache.log4j.Logger.getLogger(ResourceConstraint.class);
    /** table iResource[slot] = lecture using this resource placed in the given time slot (null if empty) */
    protected Lecture[] iResource;
    private String iResourceId;
    private String iName;
    /** Number of slots per day */
    protected int iSlotsPerDay;
    /** Number of days */
    protected int iDays;
    
    /** Constructor
     * @param id resource id
     * @param name resource name
     * @param slotsPerDay number of slots per day
     * @param days number of days
     */
    public ResourceConstraint(String id, String name, int slotsPerDay, int days) {
        iSlotsPerDay = slotsPerDay;
        iDays = days;
        iResourceId = id;
        iName = name;
        iResource = new Lecture[iSlotsPerDay * iDays];
        for (int i=0;i<iResource.length;i++)
            iResource[i]=null;
    }
    
    /** Resource id */
    public String getResourceId() { return iResourceId; }
    /** Resource name */
    public String getName() { return iName; }
    /** Number of slots per day */
    public int getSlotsPerDay() { return iSlotsPerDay; }
    /** Number of days */
    public int getDays() { return iDays; }
    
    public void computeConflicts(Value value, Set conflicts) {
        Lecture lecture = (Lecture) value.variable();
        Placement placement = (Placement) value;
        for (int i=0; i<placement.getTimeLocation().getSlots().length; i++) {
            int slot = placement.getTimeLocation().getSlots()[i];
            Lecture conf = iResource[slot];
            if (conf!=null && !lecture.equals(conf)) conflicts.add(conf.getAssignment());
        }
    }
    
    public boolean inConflict(Value value) {
        Placement placement = (Placement) value;
        for (int i=0; i<placement.getTimeLocation().getSlots().length; i++) {
            int slot = placement.getTimeLocation().getSlots()[i];
            if (iResource[slot]!=null) return true;
        }
        return false;
    }
        
    public boolean isConsistent(Value value1, Value value2) {
        Placement p1 = (Placement) value1;
        Placement p2 = (Placement) value2;
        return !p1.getTimeLocation().hasIntersection(p2.getTimeLocation());
    }
    
    public void assigned(long iteration, Value value) {
        super.assigned(iteration, value);
        Lecture lecture = (Lecture) value.variable();
        Placement placement = (Placement) value;
        for (int i=0; i<placement.getTimeLocation().getSlots().length; i++) {
            int slot = placement.getTimeLocation().getSlots()[i];
            iResource[slot] = lecture;
        }
    }
    
    public void unassigned(long iteration, Value value) {
        super.unassigned(iteration, value);
        Lecture lecture = (Lecture) value.variable();
        Placement placement = (Placement) value;
        for (int i=0; i<placement.getTimeLocation().getSlots().length; i++) {
            int slot = placement.getTimeLocation().getSlots()[i];
            iResource[slot] = null;
        }
    }
    
    /** Lookup table getResource()[slot] -> lecture using this resource placed in the given time slot (null if empty) */
    public Lecture[] getResource() { return iResource; }
    
    /** Number of useless slots for this resource */
    public int countUselessSlots() {
        int ret = 0;
        for (int d=0;d<iDays;d++) {
            for (int s=1;s<iSlotsPerDay-1;s++) {
                int slot = d*iSlotsPerDay+s;
                if (iResource[slot-1]!=null && iResource[slot]==null && iResource[slot+1]!=null)
                    ret++;
            }
        }
        return ret;
    }
    
    /** Resource usage usage */
    protected void printUsage(StringBuffer sb) {
        for (int slot=0;slot<getResource().length;slot++) {
            if (getResource()[slot]!=null) {
                int day = slot / edu.purdue.smas.timetable.util.Constants.SLOTS_PER_DAY;
                int time = (7*60)+30 + (30 * (slot % edu.purdue.smas.timetable.util.Constants.SLOTS_PER_DAY));
                int h = time / 60;
                int m = time % 60;
                String d = edu.purdue.smas.timetable.util.Constants.DAY_NAMES_SHORT[day];
                int slots = ((Placement)((Lecture)getResource()[slot]).getAssignment()).getTimeLocation().getLength();
                time += (30*slots);
                int h2 = time / 60;
                int m2 = time % 60;
                sb.append(sb.length()==0?"":",\n        ").append("["+d+(h>12?h-12:h)+":"+(m<10?"0":"")+m+(h>=12?"pm":"am")+"-"+(h2>12?h2-12:h2)+":"+(m2<10?"0":"")+m2+(h2>=12?"pm":"am")+"]=").append(((Lecture)iResource[slot]).getName());
                slot+=slots-1;
                //sb.append(sb.length()==0?"":", ").append("s"+(slot+1)+"=").append(((Lecture)getResource()[slot]).getName());
            }
        }
    }

    public String toString() {
        StringBuffer sb = new StringBuffer("Resource{name="+getName());
        printUsage(sb);
        sb.append("\n      }");
        return sb.toString();
    }    
}
