/*
 * ResourceGroup.java
 *
 * Created on 12. jen 2000, 10:48
 * Documented 7.7.2001
 */

package timetable.data;

import java.util.Vector;
import timetable.util.*;
/** Reprezantace skupiny zdroj v rozvrhovacm problmu.
 *
 * @author Tom Mller
 * @version 1.0
 */
public class ResourceGroup extends java.lang.Object { 
    /** Reprezentace zdroj pomoc tdy java.util.Vector */
    protected Vector resources = new Vector();
    
    /** Zkratka jmna skupiny zdroj*/
    public String shortCut=null;
    /** Poznmka*/
    public String note=null;
    /** Jmno*/
    public String name=null;
    /** Pznak zmny*/
    public boolean changed=false;
    /** Pznak konjunktivn i disjunktivn skupiny zdroj */
    public boolean conjunctive=false;
    
    /** Konstruktor. */
    public ResourceGroup() {
    }
    
    /** Konstruktor.
     * @param name jmno skupiny
     */
    public ResourceGroup(String name) {
        this.name=name;
    }
    
    /** Konstruktor.
     * @param resource Zdroj resource bude pidn do skupiny po jejm vytvoen.
     */
    public ResourceGroup(Resource resource) {
        resources.add(resource);
    }
    
    /** Konstruktor.
     * @param resource skupina zdroj bude pidna do skupiny po jejm vytvoen.
     */
    public ResourceGroup(Resource[] resource) {
        for (int i=0; i<resource.length;i++)
            resources.add(resource[i]);
    }
    
    /** Nastaven jmna.
     * @param name jmno
     */
    public void setName(String name) {
        this.name=name;
        changed=true;
    }
    
    /** Zskn jmna skupiny
     * @return jmno skupiny
     */
    public String getName() { return name;}
    
    /** Pidn zdroje do skupiny
     * @param zdroj
     */
    public void add(Resource resource) {
        resources.add(resource);
        changed=true;
    }
    
    /** Odebrn zdroje ze skupiny
     * @param zdroj
     */
    public void remove(Resource resource) {
        resources.remove(resource);
        changed=true;
    }
    
    /** Dotaz na zmnu skupiny
     * @return true, pokud byla zmna
     */
    public boolean isChanged() { 
        return changed; 
    }
    
    /** Dotaz na ptomnost zdroje ve skupin
     * @param resource zdroj
     * @return true, pokud je zdroj ve skupin
     */
    public boolean isInGroup(Resource resource) {
        for (int i=0;i<size();i++) 
            if (get(i)==resource) return true;
        return false;
    }
    
    /** Odebrn vech zdroj ze skupiny
     */
    public void removeAll() {
        resources.clear();
    }
    
    /** Vrt poet zdroj ve skupin
     * @return poet zdroj ve skupin
     */
    public int size() {
        return resources.size();
    }
    
    /** Vrt zdroj s poadovanm indexem. Tento index je slovn od 0.
     * @param i index zdroje ve skupin
     * @return zdroj s indexem i 
     */ 
    public Resource get(int i) {
        try {
            return (Resource) resources.get(i);
        } catch (java.lang.ArrayIndexOutOfBoundsException e) {
            return null;
        }
    }
    
    /** Nastaven zdroje s indexem
     * @param i index zdroje ve skupin
     * @param resource zdroj, kter bude na toto mstu umstn
     */
    public void set(int i,Resource resource) {
        resources.set(i,resource);
    }
    
    /** Vrt index poadovanho zdroje.
     * @param resource zdroj
     * @return index zdroje ve skupin, -1 pokud tam nen
     */
    public int indexOf(Resource resource) {
        return resources.indexOf(resource);
    }
    
    /** Otzka, zda-li je jeden ze zdroj ve skupin v poadovan as voln.
     * @param slot poadovan asov slot
     * @param length dlka aktivity, pro kterou se dan operace zjiuje -> jsou prochzeny sloty <code>slot .. slot+lengt-1</code>
     * @return true, pokud takov zdroj ve skupin je
     */
    public boolean isOneFreeAt(int slot, int length) {
        for (int i=0;i<size();i++) 
            if (get(i).isFreeAt(slot,length)) return true;
        return false;
    }
    
    /** Otzka, zda-li jsou vechny zdroje ve skupin v poadovan as voln.
     * @param slot poadovan asov slot
     * @param length dlka aktivity, pro kterou se dan operace zjiuje -> jsou prochzeny sloty <code>slot .. slot+lengt-1</code>
     * @return true, pokud jsou voln
     */
    public boolean isAllFreeAt(int slot, int length) {
        for (int i=0;i<size();i++)
            if (!get(i).isFreeAt(slot,length)) return false;
        return true;
    }

    /** Otzka, zda-li me bt jeden ze zdroj ve skupin v poadovan as voln. To znamen, e dan sloty nemaj HARD podmnku a ppadn aktivity v nich umstn jdou odebrat.
     * @param slot poadovan asov slot
     * @param length dlka aktivity, pro kterou se dan operace zjiuje -> jsou prochzeny sloty <code>slot .. slot+lengt-1</code>
     * @return true, pokud takov zdroj ve skupin je
     */
    public boolean canOneBeFreeAt(int slot, int length) {
        for (int i=0;i<size();i++) 
            if (get(i).canBeFreeAt(slot,length)) return true;
        return false;
    }

    /** Otzka, zda-li me bt vechny zdroje ve skupin v poadovan as voln. To znamen, e dan sloty nemaj HARD podmnku a ppadn aktivity v nich umstn jdou odebrat.
     * @param slot poadovan asov slot
     * @param length dlka aktivity, pro kterou se dan operace zjiuje -> jsou prochzeny sloty <code>slot .. slot+lengt-1</code>
     * @return true, pokud ano
     */
    public boolean canAllBeFreeAt(int slot, int length) {
        for (int i=0;i<size();i++) 
            if (!get(i).canBeFreeAt(slot,length)) return false;
        return true;
    }

    /** Spote poet volnch zdroj v dan as ve skupin.
     * @param slot poadovan asov slot
     * @param length dlka aktivity, pro kterou se dan operace zjiuje -> jsou prochzeny sloty <code>slot .. slot+lengt-1</code>
     * @return poet volnch zdroj
     */
    public int countFreeAt(int slot, int length) {
        int sum=0;
        for (int i=0;i<size();i++) 
            if (get(i).isFreeAt(slot,length)) sum++;
        return sum;
    }

    /** Vrt skupinu volnch zdroj v dan as
     * @param slot poadovan asov slot
     * @param length dlka aktivity, pro kterou se dan operace zjiuje -> jsou prochzeny sloty <code>slot .. slot+lengt-1</code>
     * @return skupina volnch zdroj
     */
    public ResourceGroup getFreeAt(int slot, int length) {
        ResourceGroup g = new ResourceGroup();
        boolean free=true;
        for (int i=0; i<this.size();i++) 
            if (get(i).isFreeAt(slot,length)) g.add(get(i));
        return g;
    }
    
    /** Vrt jinou instanci tto skupiny. 
     * @return jin instance stejn skupiny
     */
    public Object clone() {
        ResourceGroup g = new ResourceGroup();
        g.setName(getName());
        g.note=this.note;
        g.shortCut=this.shortCut;
        g.conjunctive=this.conjunctive;
        for (int i=0;i<size();i++) 
            g.add(get(i));
        return g;
    }    
    
    /** Porovn na rovnost dv skupiny
     * @param o druh skupina
     * @return true, pokud obsahuj stejn zdroje (ve stejnm poad) a maj stejn parametry
     */
    public boolean equals(Object o) {
        if (o==null) return false;
        if (o instanceof ResourceGroup) {
            ResourceGroup g = (ResourceGroup) o;
            if (shortCut==null && shortCut!=g.shortCut) return false;
            if (shortCut!=null && !shortCut.equals(g.shortCut)) return false;
            if (name==null && name!=g.name) return false;
            if (name!=null && !name.equals(g.name)) return false;
            if (conjunctive!=g.conjunctive) return false;
            if (size()!=g.size()) return false;
            for (int i=0;i<size();i++) 
                if (g.get(i)!=get(i)) return false;
            return true;
        } else return false;
    }
    
    /** Zjist, zda-li dv skupiny obsahuj stejn zdroje.
     * @param r druh skupina
     * @return true, pokud ob skupiny obsahuj stejn zdroje (i v jinm poad)
     */
    public boolean haveSameContent(ResourceGroup r) {
        if (r==null) return false;
        if (r.size()!=size()) return false;
        for (int i=0;i<size();i++)
            if (r.indexOf(get(i))<0) return false;
        return true;
    }
    
    /** Zpis skupiny do streamu.
     * @param out stream
     * @param allActivities mnoina vech aktivit
     */
    protected void write(java.io.ObjectOutputStream out, ActivityGroup allActivities) throws java.io.IOException {
       out.writeInt(size());
       for (int i=0;i<size();i++)
           get(i).write(out, allActivities);
   }

   /** Naten skupiny ze streamu
    * @param in stream
    * @param config konfigurace
    * @return naten skupina
    */
   protected static ResourceGroup read(java.io.ObjectInputStream in, Config config) throws java.io.IOException, java.lang.ClassNotFoundException, TimetableException {
       int nrResources=in.readInt();
       ResourceGroup g = new ResourceGroup();
       for (int i=0;i<nrResources;i++)
           g.add(Resource.read(in,config));
       return g;
   }
   
   /** Inicializace skupiny po naten 
    * @param allActivities mnoina vech aktivit
    */
   protected void init(ActivityGroup allActivities) {
       for (int i=0;i<size();i++)
           get(i).init(allActivities);
   }
   
    /** Zpis skupiny do streamu.
     * @param out stream
     * @param allResources mnoina vech zdroj
     */
   protected void write(java.io.ObjectOutputStream out, ResourceGroup allResources) throws java.io.IOException {
       out.writeObject(name);
       out.writeObject(shortCut);
       out.writeObject(note);
       out.writeBoolean(conjunctive);
       out.writeInt(size());
       for (int i=0;i<size();i++)
           get(i).write(out, allResources);
       changed=false;
   }
   
   /** Naten skupiny ze streamu
    * @param in stream
    * @param config konfigurace
    * @param allResources mnoina vech zdroj
    * @return naten skupina
    */
   protected static ResourceGroup read(java.io.ObjectInputStream in, ResourceGroup allResources) throws java.io.IOException, java.lang.ClassNotFoundException {
       String name=(String)in.readObject();
       String shortCut=(String)in.readObject();
       String note=(String)in.readObject();
       boolean conjunctive=in.readBoolean();
       int nrResources=in.readInt();
       ResourceGroup g = new ResourceGroup();
       g.name=name;g.shortCut=shortCut;g.note=note;g.conjunctive=conjunctive;
       for (int i=0;i<nrResources;i++)
           g.add(Resource.read(in,allResources));
       return g;
   }
   
   /** Vrt zkratky zdroj ve skupin jako etzec. Pklad "{Z1, Z2, Z3}"
    * @return etzec (ve sloench zvorkch rkami oddlen zkratky jmen jednotlivch zdroj)
    */
   public String toString() {
       StringBuffer b = new StringBuffer();
       b.append("{");
       for (int i=0;i<size();i++) {
           if (i!=0) b.append(", ");
           b.append(get(i).shortCut);
       };
       b.append("}");
       return b.toString();
   }
        
}
