/*
 * Resource.java
 *
 * Created on 12. jen 2000, 10:45
 * Documented 7.7.2001
 */

package timetable.data;

import timetable.util.*;

/** Reprezentace jednoho zdroje v rozvrhovacm problmu. 
 *
 * @author Tom Mller
 * @version 1.0
 */
public class Resource extends java.lang.Object {
    /** Globln konfigurace */
    protected Config config = null;
    /** Maximln dlka zkratky jmna */
    public static int SHORT_CUT_MAX_LENGTH = 7;

    /** Zkratka jmna - pro vizualizaci */
    public String shortCut = null;
    /** Jmno */
    public String name = null;
    /** Poznmka */
    public String note = null;
    /** asov preference zdroje */
    protected TimePreference preference = null;
    
    /** asov tabulka naalokovanch slot zdroje aktivitami. Slou zejmna k jednoduchmu zjitn vyuvn zdroje v dan asov okamik.*/
    protected Activity[] table = null;

    /** Indexy asov tabulky, slou pouze pro ten a zpis objektu */
    protected int[] tableIndexes = null; // For reading propose only
    
    
    /** Konstruktor.
     * @param config Globln konfigurace
     */
    public Resource(Config config) throws TimetableException {
        this.config=config;
        preference = new TimePreference(config);
        table = new Activity[config.getInt(Config.NR_SLOTS)];
        for (int i=0;i<table.length;i++) table[i]=null;
    }
    
    /** Konstruktor.
     * @param config Globln konfigurace
     * @param shortCut zkratka jmna zdroje
     * @param name jmno zdroje
     * @param note poznmka
     */
    public Resource(Config config,String shortCut,String name,String note) throws TimetableException{
        this(config);
        this.name = name;
        this.shortCut=(shortCut==null||shortCut.length()<=SHORT_CUT_MAX_LENGTH?shortCut:shortCut.substring(0,SHORT_CUT_MAX_LENGTH));
        this.note=note;
    }
    
    /** Konstruktor.
     * @param config Globln konfigurace
     * @param name jmno zdroje
     */
    public Resource(Config config,String name) throws TimetableException{
        this(config,name,name,null); 
    }
    
    /** Zskn asovch preferenc.
     * @return tabluka asovch preferenc
     */
    public TimePreference getPreference() {
        return preference;
    }
    
    /** Nastaven tabulky asovch preferenc
     * @param preference tabulka asovch preferenc
     */
    public void setPreference(TimePreference preference) {
        this.preference=preference;
    }
    
    /** Zskn aktivity, naalokovan v pslunm slotu.
     * @param slot pslun asov slot
     * @return naalokovan aktivita, <code>null</code> pokud je slot voln
     */
    public Activity getActivityAt(int slot) {
        return table[slot];
    }
    
    /** Naalokovn slotu pro danou aktivitu.
     * @param slot pslun asov slot
     * @param activity aktivta, kter bude dan zdroj v danm ase vyuvat
     */
    public void setActivityAt(int slot,Activity activity) {
        table[slot]=activity;
    }

    /** Otzka na vyuit zdroje v dan asov slot
     * @param slot dan asov slot
     * @return <code>true</code>, pokud je slot voln
     */
    public boolean isFreeAt(int slot) {
        return (!preference.isHARD(slot) && getActivityAt(slot)==null);
    }
    
    /** Otzka, zda-li me bt dan slot pouit k plnovn. Pro takov slot mus platit:
     * <ul><li>na danm slotu nen HARD preference</li>
     *<li>slot je voln nebo aktivita, kter v nm le je peplnovateln</li>
     *</ul>
     * @param slot pslun asov slot
     * @return <code>true</code> pokud lze zdroj v dan slot pout
     */
    public boolean canBeFreeAt(int slot) {
        return (!preference.isHARD(slot) && (getActivityAt(slot)==null || getActivityAt(slot).canReschedule));
    }
    
    /** Otzka, zda-li me bt dan slot a sloty nsledujc pouity k plnovn. Pro takov slot mus platit:
     * <ul><li>na danm slotu nen HARD preference</li>
     *<li>slot je voln nebo aktivita, kter v nm le je peplnovateln</li>
     *</ul>
     * @param slot pslun asov slot
     * @param length dlka aktivity, kter chce dan zdroj vyuvat -> sloty <code>slot .. slot+length-1</code> jsou testovny
     * @return <code>true</code> pokud lze zdroj v dan sloty pout
     */
    public boolean canBeFreeAt(int slot, int length) {
        for (int j=0;j<length;j++) if (!canBeFreeAt(slot+j)) return false;
        return true;
    }
    
    /** Otzka, zda-li je slot v dan asov sloty voln.
     * @param slot pslun asov slot
     * @param length dlka aktivity, kter chce dan zdroj vyuvat -> sloty <code>slot .. slot+length-1</code> jsou testovny
     * @return <code>true</code> pokud je zdroj v dan asov sloty voln
     */
    public boolean isFreeAt(int slot, int length) {
        for (int j=0;j<length;j++) if (!isFreeAt(slot+j)) return false;
        return true;
    }
    
    /** Zjiuje, zda-li je dan aktivita naplnovna konzistentn. To znamen e vechny sloty, kter vyaduje nesm mt HARD podmnku a mus bt pro danou aktivitu alokovny.
     * @param activity dan aktivita
     * @return <code>true</code>, pokud je aktivta naplnovna konzistentn (sprvn)
     */
    public boolean isConsistentFor(Activity activity) {
        for (int i=activity.start;i<activity.start+activity.length;i++) {
            if (preference.isHARD(i) || table[i]!=activity) return false;
        }
        return true;
    }
    
    /** Vrt mnoinu aktivit, kter vyuvaj dan zdroj v danch asovch slotech.
     * @param slot prvn asov slot
     * @param length dlka aktivity, tj. jsou prochzeny sloty <code>slot .. slot+length-1</code>
     * @return seznam aktivit vyuvajcch dan zdroj v dan asov sloty
     */
    public ActivityGroup getActivitiesAt(int slot, int length) {
        ActivityGroup ag = new ActivityGroup();
        for (int j=0;j<length;j++) 
            if (getActivityAt(slot+j)!=null) ag.addIfIsNot(getActivityAt(slot+j));
        return ag;
    }
    
    /** Zpis zdroje do objektovho streamu. 
     * @param out stream
     * @param allActivities seznam vech aktivit
     */
   protected void write(java.io.ObjectOutputStream out, ActivityGroup allActivities) throws java.io.IOException {
       out.writeObject(shortCut);
       out.writeObject(name);
       out.writeObject(note);
       out.writeObject(preference);
       out.writeInt(table.length);
       for (int i=0;i<table.length;i++) 
           if (getActivityAt(i)!=null) out.writeInt(allActivities.indexOf(getActivityAt(i)));
           else out.writeInt(-1);
   }

   /** Naten zdroje ze streamu.
    * @param in stream
    * @param config globln konfigurace
    * @return naten zdroj
    */
   protected static Resource read(java.io.ObjectInputStream in, Config config) throws java.io.IOException, java.lang.ClassNotFoundException,TimetableException {
       Resource z = new Resource(config);
       z.shortCut =  (String) in.readObject();
       z.name =  (String) in.readObject();
       z.note = (String) in.readObject();
       z.preference = (TimePreference) in.readObject();
       z.preference.config=config;
       z.table=null;
       z.tableIndexes = new int[in.readInt()];
       for (int i=0;i<z.tableIndexes.length;i++) z.tableIndexes[i]=in.readInt();
       return z;
   }
   
    /** Zpis zdroje do objektovho streamu. 
     * @param out stream
     * @param allResources seznam vech zdroj
     */
   protected void write(java.io.ObjectOutputStream out, ResourceGroup allResources) throws java.io.IOException {
       out.writeInt(allResources.indexOf(this));
   }

   /** Naten zdroje ze streamu.
    * @param in stream
    * @param allResources seznam vech zdroj
    * @return naten zdroj
    */
   protected static Resource read(java.io.ObjectInputStream in, ResourceGroup allResources) throws java.io.IOException {
       return allResources.get(in.readInt());
   }
   
   /** Inicializace zdroje po jeho naten*/
   protected void init(ActivityGroup allActivities) {
      if (tableIndexes==null) return;
      table = new Activity[tableIndexes.length];
      for (int i=0;i<table.length;i++) 
          if (tableIndexes[i]==-1) table[i]=null;
          else table[i]=allActivities.get(tableIndexes[i]);
      tableIndexes=null;
   }
   
   /** Vrac zkratku jmna zdroje. 
    * @return zkratka jmna zdroje 
    */
   public String toString() {
       return shortCut;
   }

}
